import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Scale, User, FileText, CheckCircle, Clock, Award, TrendingUp, Shield } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './HistoryPage.css'

const HistoryPage = () => {
  const [consultations, setConsultations] = useState([])

  useEffect(() => {
    loadConsultations()
    const interval = setInterval(loadConsultations, 500)
    return () => clearInterval(interval)
  }, [])

  const loadConsultations = () => {
    const saved = localStorage.getItem('lawConsultations')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setConsultations(parsed)
      } catch (e) {
        setConsultations([])
      }
    } else {
      const defaultConsultations = [
        {
          id: 1,
          topic: 'Семейное право',
          lawyer: 'Анна Соколова',
          status: 'Завершено',
          review: 'Отличная консультация, все разъяснили подробно'
        },
        {
          id: 2,
          topic: 'Трудовое право',
          lawyer: 'Михаил Петров',
          status: 'В процессе',
          review: null
        },
        {
          id: 3,
          topic: 'Жилищное право',
          lawyer: 'Елена Волкова',
          status: 'Завершено',
          review: 'Профессионал своего дела, рекомендую!'
        }
      ]
      setConsultations(defaultConsultations)
      localStorage.setItem('lawConsultations', JSON.stringify(defaultConsultations))
    }
  }

  const getStatusColor = (status) => {
    return status === 'Завершено' ? '#10B981' : '#F59E0B'
  }

  const totalConsultations = consultations.length
  const completedConsultations = consultations.filter(c => c.status === 'Завершено').length
  const uniqueLawyers = new Set(consultations.map(c => c.lawyer)).size

  return (
    <div className="history-page">
      <div className="container">
        <div className="page-header">
          <motion.h1 
            className="page-title"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6 }}
          >
            История обращений
          </motion.h1>
          <motion.p 
            className="page-subtitle"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            Все ваши юридические консультации
          </motion.p>
        </div>

        <motion.div 
          className="stats-minimal"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{totalConsultations}</div>
            <div className="stat-minimal-label">Обращений</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{completedConsultations}</div>
            <div className="stat-minimal-label">Завершено</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{uniqueLawyers}</div>
            <div className="stat-minimal-label">Юристов</div>
          </div>
        </motion.div>

        <ImageSlider images={[
          '/images/law-1.jpg',
          '/images/law-2.jpg',
          '/images/law-3.jpg',
          '/images/law-4.jpg'
        ]} />

        <motion.section 
          className="consultations-list"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title-minimal">Ваши обращения</h2>
          {consultations.length > 0 ? (
            <div className="consultations-vertical">
              {consultations.map((consultation, index) => (
                <motion.div 
                  key={consultation.id} 
                  className="consultation-item"
                  initial={{ opacity: 0, x: -30 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ x: 8, borderColor: 'var(--primary)' }}
                >
                  <div className="consultation-item-left">
                    <div className="consultation-item-icon">
                      <Scale size={24} strokeWidth={2} />
                    </div>
                    <div className="consultation-item-content">
                      <div className="consultation-item-header">
                        <h3 className="consultation-item-topic">{consultation.topic}</h3>
                        <div className="consultation-item-status" style={{ background: getStatusColor(consultation.status) + '20', color: getStatusColor(consultation.status) }}>
                          {consultation.status}
                        </div>
                      </div>
                      <div className="consultation-item-details">
                        <div className="consultation-item-detail">
                          <User size={16} strokeWidth={2} />
                          <span>{consultation.lawyer}</span>
                        </div>
                        {consultation.review && (
                          <div className="consultation-item-review">
                            <FileText size={16} strokeWidth={2} />
                            <span>{consultation.review}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <Scale size={64} strokeWidth={1.5} className="empty-icon" />
              <h3 className="empty-title">Нет обращений</h3>
              <p className="empty-text">Запишитесь на консультацию</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="features-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.6 }}
        >
          <h2 className="section-title-minimal">Преимущества</h2>
          <div className="features-grid">
            <div className="feature-item">
              <div className="feature-icon">
                <Shield size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Защита прав</h3>
                <p className="feature-text">Максимальная защита интересов</p>
              </div>
            </div>
            <div className="feature-item">
              <div className="feature-icon">
                <Award size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Опытные юристы</h3>
                <p className="feature-text">Специалисты с опытом</p>
              </div>
            </div>
            <div className="feature-item">
              <div className="feature-icon">
                <Clock size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Быстрая помощь</h3>
                <p className="feature-text">Консультации в удобное время</p>
              </div>
            </div>
            <div className="feature-item">
              <div className="feature-icon">
                <TrendingUp size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Высокий рейтинг</h3>
                <p className="feature-text">98% довольных клиентов</p>
              </div>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default HistoryPage

